﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/lightsail/LightsailRequest.h>
#include <aws/lightsail/Lightsail_EXPORTS.h>
#include <aws/lightsail/model/CacheBehavior.h>
#include <aws/lightsail/model/CacheBehaviorPerPath.h>
#include <aws/lightsail/model/CacheSettings.h>
#include <aws/lightsail/model/InputOrigin.h>
#include <aws/lightsail/model/IpAddressType.h>
#include <aws/lightsail/model/Tag.h>
#include <aws/lightsail/model/ViewerMinimumTlsProtocolVersionEnum.h>

#include <utility>

namespace Aws {
namespace Lightsail {
namespace Model {

/**
 */
class CreateDistributionRequest : public LightsailRequest {
 public:
  AWS_LIGHTSAIL_API CreateDistributionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateDistribution"; }

  AWS_LIGHTSAIL_API Aws::String SerializePayload() const override;

  AWS_LIGHTSAIL_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name for the distribution.</p>
   */
  inline const Aws::String& GetDistributionName() const { return m_distributionName; }
  inline bool DistributionNameHasBeenSet() const { return m_distributionNameHasBeenSet; }
  template <typename DistributionNameT = Aws::String>
  void SetDistributionName(DistributionNameT&& value) {
    m_distributionNameHasBeenSet = true;
    m_distributionName = std::forward<DistributionNameT>(value);
  }
  template <typename DistributionNameT = Aws::String>
  CreateDistributionRequest& WithDistributionName(DistributionNameT&& value) {
    SetDistributionName(std::forward<DistributionNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An object that describes the origin resource for the distribution, such as a
   * Lightsail instance, bucket, or load balancer.</p> <p>The distribution pulls,
   * caches, and serves content from the origin.</p>
   */
  inline const InputOrigin& GetOrigin() const { return m_origin; }
  inline bool OriginHasBeenSet() const { return m_originHasBeenSet; }
  template <typename OriginT = InputOrigin>
  void SetOrigin(OriginT&& value) {
    m_originHasBeenSet = true;
    m_origin = std::forward<OriginT>(value);
  }
  template <typename OriginT = InputOrigin>
  CreateDistributionRequest& WithOrigin(OriginT&& value) {
    SetOrigin(std::forward<OriginT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An object that describes the default cache behavior for the distribution.</p>
   */
  inline const CacheBehavior& GetDefaultCacheBehavior() const { return m_defaultCacheBehavior; }
  inline bool DefaultCacheBehaviorHasBeenSet() const { return m_defaultCacheBehaviorHasBeenSet; }
  template <typename DefaultCacheBehaviorT = CacheBehavior>
  void SetDefaultCacheBehavior(DefaultCacheBehaviorT&& value) {
    m_defaultCacheBehaviorHasBeenSet = true;
    m_defaultCacheBehavior = std::forward<DefaultCacheBehaviorT>(value);
  }
  template <typename DefaultCacheBehaviorT = CacheBehavior>
  CreateDistributionRequest& WithDefaultCacheBehavior(DefaultCacheBehaviorT&& value) {
    SetDefaultCacheBehavior(std::forward<DefaultCacheBehaviorT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An object that describes the cache behavior settings for the
   * distribution.</p>
   */
  inline const CacheSettings& GetCacheBehaviorSettings() const { return m_cacheBehaviorSettings; }
  inline bool CacheBehaviorSettingsHasBeenSet() const { return m_cacheBehaviorSettingsHasBeenSet; }
  template <typename CacheBehaviorSettingsT = CacheSettings>
  void SetCacheBehaviorSettings(CacheBehaviorSettingsT&& value) {
    m_cacheBehaviorSettingsHasBeenSet = true;
    m_cacheBehaviorSettings = std::forward<CacheBehaviorSettingsT>(value);
  }
  template <typename CacheBehaviorSettingsT = CacheSettings>
  CreateDistributionRequest& WithCacheBehaviorSettings(CacheBehaviorSettingsT&& value) {
    SetCacheBehaviorSettings(std::forward<CacheBehaviorSettingsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An array of objects that describe the per-path cache behavior for the
   * distribution.</p>
   */
  inline const Aws::Vector<CacheBehaviorPerPath>& GetCacheBehaviors() const { return m_cacheBehaviors; }
  inline bool CacheBehaviorsHasBeenSet() const { return m_cacheBehaviorsHasBeenSet; }
  template <typename CacheBehaviorsT = Aws::Vector<CacheBehaviorPerPath>>
  void SetCacheBehaviors(CacheBehaviorsT&& value) {
    m_cacheBehaviorsHasBeenSet = true;
    m_cacheBehaviors = std::forward<CacheBehaviorsT>(value);
  }
  template <typename CacheBehaviorsT = Aws::Vector<CacheBehaviorPerPath>>
  CreateDistributionRequest& WithCacheBehaviors(CacheBehaviorsT&& value) {
    SetCacheBehaviors(std::forward<CacheBehaviorsT>(value));
    return *this;
  }
  template <typename CacheBehaviorsT = CacheBehaviorPerPath>
  CreateDistributionRequest& AddCacheBehaviors(CacheBehaviorsT&& value) {
    m_cacheBehaviorsHasBeenSet = true;
    m_cacheBehaviors.emplace_back(std::forward<CacheBehaviorsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The bundle ID to use for the distribution.</p> <p>A distribution bundle
   * describes the specifications of your distribution, such as the monthly cost and
   * monthly network transfer quota.</p> <p>Use the
   * <code>GetDistributionBundles</code> action to get a list of distribution bundle
   * IDs that you can specify.</p>
   */
  inline const Aws::String& GetBundleId() const { return m_bundleId; }
  inline bool BundleIdHasBeenSet() const { return m_bundleIdHasBeenSet; }
  template <typename BundleIdT = Aws::String>
  void SetBundleId(BundleIdT&& value) {
    m_bundleIdHasBeenSet = true;
    m_bundleId = std::forward<BundleIdT>(value);
  }
  template <typename BundleIdT = Aws::String>
  CreateDistributionRequest& WithBundleId(BundleIdT&& value) {
    SetBundleId(std::forward<BundleIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The IP address type for the distribution.</p> <p>The possible values are
   * <code>ipv4</code> for IPv4 only, and <code>dualstack</code> for IPv4 and
   * IPv6.</p> <p>The default value is <code>dualstack</code>.</p>
   */
  inline IpAddressType GetIpAddressType() const { return m_ipAddressType; }
  inline bool IpAddressTypeHasBeenSet() const { return m_ipAddressTypeHasBeenSet; }
  inline void SetIpAddressType(IpAddressType value) {
    m_ipAddressTypeHasBeenSet = true;
    m_ipAddressType = value;
  }
  inline CreateDistributionRequest& WithIpAddressType(IpAddressType value) {
    SetIpAddressType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tag keys and optional values to add to the distribution during
   * create.</p> <p>Use the <code>TagResource</code> action to tag a resource after
   * it's created.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateDistributionRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateDistributionRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the SSL/TLS certificate that you want to attach to the
   * distribution.</p> <p>Use the <a
   * href="https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetCertificates.html">GetCertificates</a>
   * action to get a list of certificate names that you can specify.</p>
   */
  inline const Aws::String& GetCertificateName() const { return m_certificateName; }
  inline bool CertificateNameHasBeenSet() const { return m_certificateNameHasBeenSet; }
  template <typename CertificateNameT = Aws::String>
  void SetCertificateName(CertificateNameT&& value) {
    m_certificateNameHasBeenSet = true;
    m_certificateName = std::forward<CertificateNameT>(value);
  }
  template <typename CertificateNameT = Aws::String>
  CreateDistributionRequest& WithCertificateName(CertificateNameT&& value) {
    SetCertificateName(std::forward<CertificateNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The minimum TLS protocol version for the SSL/TLS certificate.</p>
   */
  inline ViewerMinimumTlsProtocolVersionEnum GetViewerMinimumTlsProtocolVersion() const { return m_viewerMinimumTlsProtocolVersion; }
  inline bool ViewerMinimumTlsProtocolVersionHasBeenSet() const { return m_viewerMinimumTlsProtocolVersionHasBeenSet; }
  inline void SetViewerMinimumTlsProtocolVersion(ViewerMinimumTlsProtocolVersionEnum value) {
    m_viewerMinimumTlsProtocolVersionHasBeenSet = true;
    m_viewerMinimumTlsProtocolVersion = value;
  }
  inline CreateDistributionRequest& WithViewerMinimumTlsProtocolVersion(ViewerMinimumTlsProtocolVersionEnum value) {
    SetViewerMinimumTlsProtocolVersion(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_distributionName;
  bool m_distributionNameHasBeenSet = false;

  InputOrigin m_origin;
  bool m_originHasBeenSet = false;

  CacheBehavior m_defaultCacheBehavior;
  bool m_defaultCacheBehaviorHasBeenSet = false;

  CacheSettings m_cacheBehaviorSettings;
  bool m_cacheBehaviorSettingsHasBeenSet = false;

  Aws::Vector<CacheBehaviorPerPath> m_cacheBehaviors;
  bool m_cacheBehaviorsHasBeenSet = false;

  Aws::String m_bundleId;
  bool m_bundleIdHasBeenSet = false;

  IpAddressType m_ipAddressType{IpAddressType::NOT_SET};
  bool m_ipAddressTypeHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;

  Aws::String m_certificateName;
  bool m_certificateNameHasBeenSet = false;

  ViewerMinimumTlsProtocolVersionEnum m_viewerMinimumTlsProtocolVersion{ViewerMinimumTlsProtocolVersionEnum::NOT_SET};
  bool m_viewerMinimumTlsProtocolVersionHasBeenSet = false;
};

}  // namespace Model
}  // namespace Lightsail
}  // namespace Aws
